// File reader service for reading the datasets and sequence files. It tries to
// recognize the file type and read the relevant parts of it.  
//
// MTZ file: 	read the column labels and types 
//
// SCA/hkl/HKL file:	read the cell parameters and spacegroup, if available 
// 
// SEQ/PIR/FASTA file: do nothing for now
//
// The result of the reading is returned in an object.
//
// JavaScript here is based on this tutorial: 
// http://odetocode.com/blogs/scott/archive/2013/07/03/building-a-filereader-service-for-angularjs-the-service.aspx
(function(module) {

	var shelxReader = function($q, $log) {

		var onLoad = function(reader, deferred, scope, fileinfo) {
			return function() {
				var result = {};
				if (fileinfo.text) {
					var lines = reader.result.split(/[\r\n]+/g);
					result.msg = "This is a text file!";
					result.filename = fileinfo.name;
					result.ext = fileinfo.ext;
					// if this file is from XDS, then read the cell parameters
					// and spacegroup
					if (lines[0].match(/FORMAT\=XDS_ASCII/i)) {
						result = readXDS(lines, result);
						// if the previous test fails, look at the file
						// extension
					} else if (fileinfo.ext == 'hkl') {
						// in this case it's assument to be a SHELXC hkl file
						// nothing else is done for this type. The user should
						// type in the cell parameters and the spacegroup.
						result.cell = {};
						result.spacegroup = "";
						result.type = 'hkl';
					} else if (fileinfo.ext == 'sca') {
						// for sca files also read the cell parameters and
						// spacegroup
						result = readSCA(lines, result);
					}
					;
				} else if (fileinfo.isMTZ) {
					var fsize = fileinfo.size;
					result.msg = "This is an MTZ file!";
					result.filename = fileinfo.name;
					result.ext = fileinfo.ext;
					result = readMTZ(reader, result, fsize);
				}
				;
				scope.$apply(function() {
					deferred.resolve(result);
				});
			};
		};

		var onError = function(reader, deferred, scope) {
			return function() {
				scope.$apply(function() {
					deferred.reject(reader.result);
				});
			};
		};

		var onProgress = function(reader, scope) {
			return function(event) {
				scope.$broadcast("fileProgress", {
					total : event.total,
					loaded : event.loaded
				});
			};
		};

		var getReader = function(deferred, scope, fileinfo) {
			var reader = new FileReader();
			reader.onload = onLoad(reader, deferred, scope, fileinfo);
			reader.onerror = onError(reader, deferred, scope);
			reader.onprogress = onProgress(reader, scope);
			return reader;
		};

		var readXDS = function(lines, result0) {
			var result = result0;
			result.type = 'XDS';
			var cell = {};
			var spacegroup = "";
			var re1 = /UNIT_CELL_CONSTANTS\=\s*([\d\.]+)\s+([\d\.]+)\s+([\d\.]+)\s+([\d\.]+)\s+([\d\.]+)\s+([\d\.]+)/i;
			//var re2 = /SPACE_GROUP_NUMBER\=\s*(\d+)/i;
			for (var i = 0; i < lines.length; i++) {
				var line = lines[i];
				var match1 = line.match(re1);
				//var match2 = line.match(re2);
				if (match1) {
					cell.a = match1[1];
					cell.b = match1[2];
					cell.c = match1[3];
					cell.alpha = match1[4];
					cell.beta = match1[5];
					cell.gamma = match1[6];
					//if (spacegroup) {
						break;
					//}
					//;
				}
				;
				//if (match2) {
				//	spacegroup = match2[1];
				//	if (!_.isEmpty(cell)) {
				//		break;
				//	}
				//	;
				//}
				//;
			}
			;
			result.cell = cell;
			//result.spacegroup = spacegroup;
			return result;
		};

		var readSCA = function(lines, result0) {
			var result = result0;
			result.type = 'sca';
			var re = /([\d\.]+\s+)([\d\.]+)\s+([\d\.]+)\s+([\d\.]+)\s+([\d\.]+)\s+([\d\.]+)\s+([cpifr]\d+)/i;
			var match = lines[2].match(re);
			var cell = {};
			var spacegroup = "";
			if (match) {
				cell.a = match[1];
				cell.b = match[2];
				cell.c = match[3];
				cell.alpha = match[4];
				cell.beta = match[5];
				cell.gamma = match[6];
				spacegroup = match[7];
				failed = false;
			}
			;
			result.cell = cell;
			result.spacegroup = spacegroup;
			return result;
		};

		var readMTZ = function(reader, result0, fsize) {
			var result = result0;
			result.type = 'mtz';
			var columns = [];
			var columnTypes = [];
			var failed = false;
			var buffer = reader.result;
			var dataView = new DataView(buffer);
			// make sure that it's an MTZ file
			var first3 = "";
			for (var i = 0; i < 3; i++) {
				try {
					first3 += String.fromCharCode(dataView.getInt8(i, false));
				} catch (err) {
					break;
				}
			}
			;
			if (first3 != 'MTZ') {
				failed = true;
			}
			;
			if (!failed) {
				try {
					var headerPos = dataView.getInt32(4, true); // little endian
					if (headerPos < 1 || headerPos > fsize / 4) {
						headerPos = dataView.getInt32(4, false); // big
																	// endian
					}
					;
					headerPos = headerPos * 4 - 1;
					var header = "";
					for (var i = headerPos; i < fsize; ++i) {
						header += String.fromCharCode(dataView
								.getInt8(i, false));
					}
					;
					var headers = header.split(/\s+/);
					var nh = headers.length;
					for (var i = 0; i < nh; i++) {
						if (/.*COLUMN.*/.test(headers[i])) {
							columns.push(headers[i + 1]);
							columnTypes.push(headers[i + 2]);
						}
						;
					}
					;
				} catch (err) {
					console.log(err);
					failed = true;
				}
				;
			}
			;
			result.columns = columns;
			result.columnTypes = columnTypes;
			result.failed = failed;
			return result;
		};

		var readFile = function(file, scope, type) {
			var fname = file.name;
			var fsize = file.size;
			var ext = fname.split('.').pop();
			var text = true;
			var isMTZ = false;
			if (ext.toLowerCase() == 'mtz') {
				text = false;
				isMTZ = true;
			}
			;
			var fileinfo = {
				name : fname,
				size : fsize,
				type : type,
				text : text,
				isMTZ : isMTZ,
				ext : ext
			};
			var deferred = $q.defer();
			var reader = getReader(deferred, scope, fileinfo);
			if (text) {
				reader.readAsText(file);
			} else {
				reader.readAsArrayBuffer(file);
			}
			return deferred.promise;
		};

		// characters that need to be escaped
		var escapeRegExp = function(string) {
			return string.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");
		}

		var sortColumnLabels = function(labels, tag) {
			var re = new RegExp(".*" + escapeRegExp(tag) + ".*");
			return _.sortBy(labels, function(label) {
				if (re.test(label)) {
					return 0;
				} else {
					return 1;
				}
			});
		}

		return {
			readFile : readFile,
			sortColumnLabels : sortColumnLabels
		};
		
	};

	module.factory('shelxReader', [ '$q', '$log', shelxReader ]);

}(angular.module('ShelxApp')));
